/***************************************************************************
* @brief NXP LPC15xx I2C example 
*	
*	Description : 
*					Driver for the I2C hardware interface on the LPC15xx
*  				A transfer consists of one or more linked messages separated by
*  				repeated STARTs. To start a transfer call function "I2C_Transfer"
*
* @note
* Copyright(C) NXP Semiconductors, 2014
* All rights reserved.
*
* Software that is described herein is for illustrative purposes only
* which provides customers with programming information regarding the
* LPC products.  This software is supplied "AS IS" without any warranties of
* any kind, and NXP Semiconductors and its licensor disclaim any and
* all warranties, express or implied, including all implied warranties of
* merchantability, fitness for a particular purpose and non-infringement of
* intellectual property rights.  NXP Semiconductors assumes no responsibility
* or liability for the use of the software, conveys no license or rights under any
* patent, copyright, mask work right, or any other intellectual property rights in
* or to any products. NXP Semiconductors reserves the right to make changes
* in the software without notification. NXP Semiconductors also makes no
* representation or warranty that such application will be suitable for the
* specified use without further testing or modification.
*
* Permission to use, copy, modify, and distribute this software and its
* documentation is hereby granted, under NXP Semiconductors' and its
* licensor's relevant copyrights in the software, without fee, provided that it
* is used in conjunction with NXP Semiconductors microcontrollers.  This
* copyright, permission, and disclaimer notice must appear in all copies of
* this code.
**********************************************************************************/
#include "LPC15xx.h"                               // LPC15xx definitions
#include "i2c.h"

volatile static I2C_MSG *msg;                               // ptr to active message block
volatile static uint8_t  cnt;                               // bytes send/received counter
volatile static uint8_t  state;                             // driver state

void I2C0_IRQHandler(void)
/***********************/
{
  uint32_t status = LPC_I2C->I2CINTSTAT;

    if (status & (1 << 24))                        // event time-out?
    {
        LPC_I2C->I2CSTAT = (1 << 24);              // clear flag
        state = I2C_TIME_OUT;
    }
    else if (status & (1 << 4))                    // master arbitration lost?
    {
        LPC_I2C->I2CSTAT = (1 << 4);
        state = I2C_ARBITRATION_LOST;
    }
    else if (status & (1 << 6))                    // master start/stop error?
    {
        LPC_I2C->I2CSTAT = (1 << 6);
        state = I2C_ERR;
    }
    else if (status & (1 << 0))                    // master pending?
    {
        switch (LPC_I2C->I2CSTAT & 0xE)            // only check master states
        {
          case (0 << 1):                           // Idle
            LPC_I2C->I2CINTENCLR = (1 << 0);       // disable master pending interrupt
            break;

/* RX */  case (1 << 1):                           // SLA+R transmitted ACK received or DATA received ACK returned
            msg->buf[cnt++] = LPC_I2C->MSTDAT;     // read data
            if (cnt != msg->nrBytes)               // last data?
            {
                LPC_I2C->MSTCTL = 0x00000001;      // no, master continue
            }
            else if (msg->next != 0)               // yes, any more messages?
            {
                cnt = 0;
                msg = (I2C_MSG *) msg->next;       // yes, point to next message
                LPC_I2C->MSTDAT = msg->address;
                LPC_I2C->MSTCTL = 0x00000003;      // generate (rep)START and continue
            }
            else                                   // ready
            {
                LPC_I2C->MSTCTL = 0x00000005;      // generate STOP and continue
                state = I2C_OK;
            }
            break;
	  
/* TX */  case (2 << 1):                           // SLA+W or DATA transmitted, ACK received
            if (cnt < msg->nrBytes)                // DATA, REP_START or STOP will be transmitted
            {
                LPC_I2C->MSTDAT = msg->buf[cnt++]; // sent byte
                LPC_I2C->MSTCTL = 0x00000001;      // master continue
            }
            else if (msg->next != 0)               // any more messages to send?
            {
                cnt = 0;
                msg = (I2C_MSG *) msg->next;       // next message
                LPC_I2C->MSTDAT = msg->address;
                LPC_I2C->MSTCTL = 0x00000003;      // generate (rep)START and continue
            }
            else                                   // ready
            {
                LPC_I2C->MSTCTL = 0x00000005;      // generate STOP and continue
                state = I2C_OK;
            }
            break;
	  
          case (3 << 1):                           // no ack on address
            LPC_I2C->MSTCTL = 0x00000005;          // generate STOP do not continue
            state = I2C_NACK_ON_ADDRESS;
            break;
	  
          case (4 << 1):                           // no ack on data
            LPC_I2C->MSTCTL = 0x00000005;          // generate STOP and continue
            state = I2C_NACK_ON_DATA;
            break;

          default:
            break;
        }	  
    }
}
	
void I2CMST_Init(void)
/********************/
{
#define SystemCoreClock          72000000

    LPC_SWM->PINENABLE1 &= ~(1 << 3);                      // enable I2C SDA pin at P0_23
    LPC_SWM->PINENABLE1 &= ~(1 << 4);                      // enable I2C SCL pin at P0_22

    LPC_SYSCON->SYSAHBCLKCTRL1 |= (EN1_I2C);               // I2C clock enable

    LPC_I2C->I2CTIMEOUT = 0x0000008F;                      // time-out of 8 x 16 I2C clock counts
  
    LPC_I2C->I2CCLKDIV = (SystemCoreClock / 2000000)-1;    // I2C clock = 2 MHz (500 nsec)
    LPC_I2C->MSTTIME   = 0;                                // SCL_low = SCL_high = 2 clocks = 1usec (1000kb/s)

    LPC_I2C->I2CINTENSET |= (1 << 4)  |                    // enable arbitration lost interrupt
                            (1 << 6)  |                    // enable master start/stop error interrupt
                            (1 << 24);                     // enable event time-out interrupt

    NVIC_EnableIRQ(I2C_IRQn);                              // enable I2C interrupt
    LPC_I2C->I2CCFG |= (1<<0) | (1<<3);                    // enable master and time-out function
}

uint8_t I2C_Transfer(I2C_MSG *p)
/*******************************
 * Start an I2C transfer, containing 1 or more LINKED messages.
 * The application must leave the message parameter block untouched.
 * The first I2C message is started with sending a START condition.
 ***************************************************************************/
{
    while (!(LPC_I2C->I2CSTAT & 1)) ;                      // wait if (previous) still pending

    msg = p;
    cnt = 0;
    state = I2C_BUSY;                                      // state of the I2C driver: BUSY

    LPC_I2C->MSTDAT      = msg->address;                   // Slave address + R/W
    LPC_I2C->MSTCTL      = 0x00000003;                     // generate START
    LPC_I2C->I2CINTENSET = (1 << 0);                       // enable master pending interrupt

    while (state == I2C_BUSY) ;                            // wait for transfer to end

    return state;
}
